//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3
import QtGraphicalEffects 1.0

import wesual.Ui 1.0

import wesual.data.Core        1.0
import wesual.data.FoodService 1.0

Rectangle {
    id : addAllergen

    property FoodServiceAllergenFilter filter
    property Document currentProduct
    property var      allergenListing
    property color    baseColor     : UiColors.getColor(UiColors.White)
    property int      maximumWidth
    property bool     popupOpen     : allergenFlyout.popupVisible
    property bool     disableFlyout : false

    implicitHeight : 40
    implicitWidth  : row.implicitWidth ? row.implicitWidth + 10 : icon.width + 10

    QtObject {
        id : p_

        readonly property int allergenSize : 34
    }

    readonly property int numberOfShownAllergens :
                     Math.max(1, Math.round(maximumWidth / p_.allergenSize)) - 1

    signal clicked()

    function showFlyout() {
        if (!allergenFlyout.popupVisible) {
            allergenFlyout.show();
            allergenFlyout.popupItem.allergenListing = allergenListing;
            allergenFlyout.popupItem.gridViewFocus   = true;
        }
    }

    activeFocusOnTab : true

    // Key handling
    Keys.onEnterPressed  : showFlyout()
    Keys.onReturnPressed : showFlyout()
    Keys.onSpacePressed  : showFlyout()

    states : [
        State {
            name : "disabledFlyout"
            when : disableFlyout
            PropertyChanges {
                target : addAllergen
                color  : addAllergen.baseColor
            }
        },
        State {
            name : "selected"
            when : mouseArea.pressed || allergenFlyout.popupVisible

            changes : [
                PropertyChanges {
                    target : addAllergen
                    color  : UiColors.getColor(UiColors.SelectionGreen)
                }
            ]
        },
        State {
            name : "hovered"
            when : mouseArea.containsMouse || addAllergen.activeFocus

            PropertyChanges {
                target : addAllergen
                color  : UiColors.getColor(UiColors.HoverGreen)
            }
        }
    ]

    color  : baseColor
    width  : row.implicitWidth ? row.implicitWidth + 10 : icon.width + 10
    height : 40

    UiColorizedImage {
        id : icon

        width  : 37
        height : 26

        sourceSize {
            width  : 37
            height : 26
        }

        color  : UiColors.getColor(UiColors.DarkGrey80)
        source : "qrc:/data_foodservice/icons/more_options.png"

        anchors.centerIn : parent

        visible : !disableFlyout &&
                  ((currentProduct && currentProduct.allergens.length === 0) ||
                  (filter && filter.tags.length === 0))

        states : State {
            name : "highlight"
            when : allergenFlyout.popupVisible

            PropertyChanges {
                target : icon
                color  : UiColors.getColor(UiColors.White)
            }
        }
    }

    Row {
        id : row

        spacing : 5

        anchors.centerIn : parent

        Repeater {
            id : repeater

            model : {
                if (currentProduct)
                    return currentProduct.allergens;
                else if (filter)
                    return filter.tags;
                else
                    return [];
            }
            delegate : UiColorizedImage {
                color   : UiColors.getColor(UiColors.DarkGrey80)
                source  : modelData.icon
                width   : 34
                height  : 34
                visible : index < numberOfShownAllergens
            }
        }

        Rectangle {
            id : allergenCounter

            width   : 34
            height  : 34
            radius  : width / 2
            color   : "transparent"
            visible : repeater.model ?
                      repeater.model.length > numberOfShownAllergens : false

            border.color : UiColors.getColor(UiColors.DarkGrey80)
            border.width : 1

            Text {
                id : allergenCount

                text  : repeater.model ?
                       (repeater.model.length - numberOfShownAllergens) + "+" :
                       ""
                color : UiColors.getColor(UiColors.DarkGrey80)

                font.pixelSize : 15

                anchors.centerIn : parent
            }
        }
    }

    MouseArea {
        id : mouseArea

        anchors.fill : parent
        hoverEnabled : enabled
        enabled      : !disableFlyout
        cursorShape  : disableFlyout ? Qt.ArrowCursor : Qt.PointingHandCursor

        onClicked : {
            addAllergen.clicked();
            addAllergen.showFlyout();
        }
    }

    FoodServiceAllergenFlyout {
        id : allergenFlyout

        triggerItem : mouseArea
        currentProduct : addAllergen.currentProduct
        filter : addAllergen.filter
    }
}
